<?php
if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly


/**
 * WAS Shipping Zone condition.
 *
 * Add the shipping zone condition to Advanced Shipping.
 *
 * @since 1.1.0
 */
class WAS_Condition_Shipping_Zone {


	/**
	 * Constructor.
	 *
	 * @since 1.1.0
	 */
	public function __construct() {

		// Add zone to conditions list
		add_filter( 'was_conditions', array( $this, 'was_conditions_add_zones' ), 10, 1 );

		// Add zone to values list
		add_filter( 'was_values', array( $this, 'was_values_add_zones' ), 10, 2 );

		// Add description to zone condition
		add_filter( 'was_descriptions', array( $this, 'was_descriptions_zones' ) );

		// Match zones
		add_filter( 'was_match_condition_zones', array( $this, 'was_match_condition_zones' ), 10, 3 );

	}


	/**
	 * Zone condition.
	 *
	 * Add the shipping zone to the conditions list.
	 *
	 * @since 1.0.0
	 *
	 * @param 	array $conditions 	List existing of conditions.
	 * @return 	array				List of modified conditions including zones.
	 */
	public function was_conditions_add_zones( $conditions ) {

		$conditions['General']['zones'] = __( 'Shipping zone', 'woocommerce-advanced-shipping' );

		return $conditions;

	}


	/**
	 * Zone value.
	 *
	 * Add zones to the condition values.
	 *
	 * @since 1.0.0
	 *
	 * @param 	array $values 		Current values (empty).
	 * @param 	array $condition	Current condition.
	 * @return 	array				Values.
	 */
	public function was_values_add_zones( $values, $condition ) {

		switch ( $condition ) {

			case 'zones':

				$values['field'] 		= 'select';
				$values['options'] 		= array();

				$zone_args = array(
					'posts_per_page'	=> '-1',
					'post_type'			=> 'shipping_zone',
				);
				$zones = get_posts( $zone_args );
				foreach ( $zones as $zone ) {
					$values['options'][ $zone->ID ] = $zone->post_title;
				}

				break;

		}

		return $values;

	}


	/**
	 * Zone description.
	 *
	 * Add a description for the shipping zones.
	 *
	 * @since 1.0.0
	 *
	 * @param 	array $descriptions List existing of descriptions.
	 * @return 	array				List of modified descriptions including zones.
	 */
	public function was_descriptions_zones( $descriptions ) {

		$descriptions['zones'] = sprintf( __( 'Zones can be added through the \'Shipping zones\' page', 'woocommerce-advanced-shipping' ), '' );

		return $descriptions;

	}


	/**
	 * Must match given zone.
	 *
	 * @param bool $match
	 * @param string $operator
	 * @param mixed $value
	 * @return bool
	 */
	public function was_match_condition_zones( $match, $operator, $value ) {

		if ( ! isset( WC()->customer ) ) {
			return;
		}

		$zone = get_post( $value );

		// Stop if post is invalid or not shipping_zone
		if ( ! $zone || 'shipping_zone' != $zone->post_type  ) {
			return;
		}

		$country_match 	= false;
		$state_match 	= false;
		$zipcode_match 	= false;

		$countries 	= (array) get_post_meta( $zone->ID, '_countries', true );
		$states	 	= (array) get_post_meta( $zone->ID, '_states', true );
		$zipcodes 	= (array) preg_split('/[\n\,]+/', get_post_meta( $zone->ID, '_zipcodes', true ) );

		// Remove all non- letters and numbers
		foreach ( $zipcodes as $key => $zipcode ) {
			$zip = preg_replace( '/[^0-9a-zA-Z\-]/', '', $zipcode );
			$zipcodes[ $key ] = strtoupper( $zip );
		}

		$user_country 	= WC()->customer->get_shipping_country();
		$user_state 	= WC()->customer->get_shipping_country() . '_' . WC()->customer->get_shipping_state();
		$user_zipcode	= WC()->customer->get_shipping_postcode();
		$user_zipcode   = preg_replace( '/[^0-9a-zA-Z]/', '', $user_zipcode );
		$user_zipcode   = strtoupper( $user_zipcode );

		if ( '==' == $operator ) {

			$country_match 	= in_array( $user_country, $countries ) && ! empty( $user_country );
			$state_match 	= in_array( $user_state, $states ) && ! empty( $user_state );
			$zipcode_match 	= in_array( $user_zipcode, $zipcodes ) && ! empty( $user_zipcode );

			foreach ( $zipcodes as $zipcode ) {

				if ( empty( $zipcode ) ) {
					continue;
				}

				if ( $zipcode_match == true ) {
					break;
				}

				$parts = explode( '-', $zipcode );
				if ( count( $parts ) > 1 ) {
					$zipcode_match = ( $user_zipcode >= min( $parts ) && $user_zipcode <= max( $parts ) );
				} else {
					$zipcode_match = preg_match( "/^" . preg_quote( $zipcode, '/' ) . "/i", $user_zipcode );
				}
			}

			if ( true == $country_match || true == $state_match || true == $zipcode_match ) {
				$match = true;
			}

		} elseif ( '!=' == $operator ) {

			$match = true;

			$country_match 	= in_array( $user_country, $countries );
			$state_match 	= in_array( $user_state, $states );
			$zipcode_match 	= ! empty( $user_zipcode ) && in_array( $user_zipcode, $zipcodes );

			foreach ( $zipcodes as $zipcode ) {

				if ( empty( $zipcode ) ) {
					continue;
				}

				if ( $zipcode_match == true ) {
					break;
				}

				$parts = explode( '-', $zipcode );
				if ( count( $parts ) > 1 ) {
					$zipcode_match = ( $user_zipcode >= min( $parts ) && $user_zipcode <= max( $parts ) );
				} else {
					$zipcode_match = preg_match( "/^" . preg_quote( $zipcode ) . "/i", $user_zipcode );
				}
			}

			if ( true == $country_match || true == $state_match || true == $zipcode_match ) {
				$match = false;
			}

		}

		return $match;

	}


}
