<?php
if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * Class WASAP_Pricing_Option_Cost_Per_Weight.
 *
 * Cost per weight pricing option.
 *
 * @class       WASAP_Pricing_Option_Cost_Per_Weight
 * @author     	Jeroen Sormani
 * @package		WooCommerce Advanced Shipping Advanced Pricing
 * @version		2.0.0
 */
class WASAP_Pricing_Option_Cost_Per_Weight extends WASAP_Pricing_Option_Abstract {


	/**
	 * Constructor.
	 *
	 * @since 2.0.0
	 */
	public function __construct() {

		$this->id   = 'cost_per_weight';
		$this->name = __( 'Cost per weight', 'woocommerce-advanced-shipping-advanced-pricing' );


		parent::__construct();

	}


	/**
	 * Output settings.
	 *
	 * Output the settings related to this cost option.
	 *
	 * @since 2.0.0
	 */
	public function output() {

		$pricing_rules = $this->get_pricing_rules();

		?><div class='cost-per-weight-wrap'>

			<div class='repeater-header row columns-4 span-6'>
				<div class='col'><label>
					<span class='label-text'><?php _e( 'Min weight', 'woocommerce-advanced-shipping-advanced-pricing' ); ?></span>
					<img class="help_tip" data-tip="<?php _e( 'You can set a minimum weight per row before the fee amount is applied.<br/>Leave empty to not set a minimum.', 'woocommerce-advanced-shipping-advanced-pricing' ); ?>" src="<?php echo  WC()->plugin_url(); ?>/assets/images/help.png" height="16" width="16" />
				</label></div>
				<div class='col'><label>
					<span class='label-text'><?php _e( 'Max weight', 'woocommerce-advanced-shipping-advanced-pricing' ); ?></span>
					<img class="help_tip" data-tip="<?php _e( 'You can set a maximum weight per row before the fee amount is applied.<br/>Leave empty to not set a maximum.', 'woocommerce-advanced-shipping-advanced-pricing' ); ?>" src="<?php echo  WC()->plugin_url(); ?>/assets/images/help.png" height="16" width="16" />
				</label></div>
				<div class='col'><label>
					<span class='label-text'><?php _e( 'Fee amount', 'woocommerce-advanced-shipping-advanced-pricing' ); ?></span>
					<img class="help_tip" data-tip="<?php _e( 'Options are:<br/>5 : Flat amount<br/>5% : percentage of subtotal<br/>5%% : percentage of product total.<br/>Add a minus (-) to deduct.<br/>Add a asterisk (*) to multiply with the weight amount.', 'woocommerce-advanced-shipping-advanced-pricing' ); ?>" src="<?php echo  WC()->plugin_url(); ?>/assets/images/help.png" height="16" width="16" />
				</label></div>
				<div class='col-inline'></div>
			</div>
			<div class='repeater-wrap'><?php

				$i = 0;
				if ( is_array( $pricing_rules ) ) :
					foreach ( $pricing_rules as $values ) :

						$i++;
						$min  = isset( $values['condition']['min'] ) ? esc_attr( $values['condition']['min'] ) : '';
						$max  = isset( $values['condition']['max'] ) ? esc_attr( $values['condition']['max'] ) : '';
						$cost = isset( $values['action']['cost'] ) ? esc_attr( $values['action']['cost'] ) : '';

						?><div class='cost-per-weight-option repeater-row row columns-4 span-6'>

							<div class='col'>
								<input type='text' class='cost-per-weight-min' name='advanced_pricing_<?php echo esc_attr( $this->id ); ?>[<?php echo absint( $i ); ?>][condition][min]' value='<?php echo $min; ?>'>
							</div>
							<div class='col'>
								<input type='text' class='cost-per-weight-max' name='advanced_pricing_<?php echo esc_attr( $this->id ); ?>[<?php echo absint( $i ); ?>][condition][max]' value='<?php echo $max; ?>'>
							</div>
							<div class='col'>
								<input type='text' class='cost-per-weight-cost wasap_input_price' name='advanced_pricing_<?php echo esc_attr( $this->id ); ?>[<?php echo absint( $i ); ?>][action][cost]' value='<?php echo esc_attr( wc_format_localized_price( $cost ) ); ?>' placeholder='<?php _e( 'Fixed amount or percentage', 'woocommerce-advanced-shipping-advanced-pricing' ); ?>'>
							</div>
							<div class='col-inline'>
								<span class='dashicons dashicons-no-alt delete-repeater-row' style='line-height: 29px;'></span>
							</div>

						</div><?php

					endforeach;
				else :

					?><div class='cost-per-weight-option repeater-row row columns-4 span-6'>
						<div class='col'>
							<input type='text' class='cost-per-weight-min' name='advanced_pricing_<?php echo esc_attr( $this->id ); ?>[0][condition][min]' value=''>
						</div>
						<div class='col'>
							<input type='text' class='cost-per-weight-max' name='advanced_pricing_<?php echo esc_attr( $this->id ); ?>[0][condition][max]' value=''>
						</div>
						<div class='col'>
							<input type='text' class='cost-per-weight-cost wasap_input_price' name='advanced_pricing_<?php echo esc_attr( $this->id ); ?>[0][action][cost]' value='' placeholder='<?php _e( 'Fixed amount or percentage', 'woocommerce-advanced-shipping-advanced-pricing' ); ?>'>
						</div>

						<div class='col-inline'></div>

					</div><?php

				endif;

			?></div>

			<a href='javascript:void(0);' class='button secondary-button add add-repeat-row'><?php _e( 'Add new', 'woocommerce-advanced-shipping-advanced-pricing' ); ?></a>

		</div><?php

		// Add new repeater row
		wc_enqueue_js( "
			jQuery( '.wasap-tabbed-settings #cost_per_weight ' ).on( 'click', '.add-repeat-row', function() {

				var repeater_wrap = $( this ).prev( '.repeater-wrap' );
				var clone = repeater_wrap.find( '.repeater-row' ).first().clone();
				var repeater_index = repeater_wrap.find( '.repeater-row' ).length;
				repeater_index++;
				clone.find( '[name*=\"[condition][min]\"]' ).attr( 'name', 'advanced_pricing_cost_per_weight[' + parseInt( repeater_index ) + '][condition][min]' ).val( '' );
				clone.find( '[name*=\"[condition][max]\"]' ).attr( 'name', 'advanced_pricing_cost_per_weight[' + parseInt( repeater_index ) + '][condition][max]' ).val( '' );
				clone.find( '[name*=\"[action][cost]\"]' ).attr( 'name', 'advanced_pricing_cost_per_weight[' + parseInt( repeater_index ) + '][action][cost]' ).val( '' );
				repeater_wrap.append( clone ).find( '.repeater-row' ).last().hide().slideDown( 'fast' );

			});
		" );

	}


	/**
	 * Get Cost.
	 *
	 * Apply the advanced pricing cost to the shipping cost.
	 *
	 * @since 2.0.0
	 *
	 * @param  float $shipping_cost    Current shipping cost.
	 * @param  int   $shipping_rate_id ID of the shipping rate being calculated.
	 * @param  array $package          Shipping package being processed.
	 * @return float                   Modified shipping cost.
	 */
	public function apply_advanced_pricing_cost( $shipping_cost, $shipping_rate_id, $package ) {

		$pricing_rules = $this->get_pricing_rules( $shipping_rate_id );
		$weight        = WC()->cart->get_cart_contents_weight();

		if ( is_array( $pricing_rules ) ) :
			foreach ( $pricing_rules as $values ) :

				$min  = isset( $values['condition']['min'] ) ? esc_attr( $values['condition']['min'] ) : null;
				$max  = isset( $values['condition']['max'] ) ? esc_attr( $values['condition']['max'] ) : null;
				$cost = isset( $values['action']['cost'] ) ? esc_attr( $values['action']['cost'] ) : null;

				// Bail if cost is not set
				if ( empty( $cost ) ) :
					continue;
				endif;

				// Bail if minimum is not set, or item qty is not met
				if ( is_null( $min ) || ( ! empty( $min ) && $weight < $min ) ) :
					continue;
				endif;

				// Bail if maximum is not set, or item qty is not met
				if ( is_null( $max ) || ( ! empty( $max ) && $weight > $max ) ) :
					continue;
				endif;

				$shipping_cost += wasap_apply_shipping_cost( $cost, $weight, 0, $min );

			endforeach;
		endif;

		return $shipping_cost;

	}


}
