<?php
if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * Class WASAP_Pricing_Option_Cost_Per_Product.
 *
 * Cost per product cost option.
 *
 * @class       WASAP_Pricing_Option_Cost_Per_Product
 * @author     	Jeroen Sormani
 * @package		WooCommerce Advanced Shipping Advanced Pricing
 * @version		2.0.0
 */
class WASAP_Pricing_Option_Cost_Per_Product extends WASAP_Pricing_Option_Abstract {


	/**
	 * Constructor.
	 *
	 * @since 2.0.0
	 */
	public function __construct() {

		$this->id   = 'cost_per_product';
		$this->name = __( 'Cost per product', 'woocommerce-advanced-shipping-advanced-pricing' );

		parent::__construct();

	}


	/**
	 * Output settings.
	 *
	 * Output the settings related to this cost option.
	 *
	 * @since 2.0.0
	 */
	public function output() {

		$post_id       = get_the_ID();
		$pricing_rules = $this->get_pricing_rules();

		?><div class='cost-per-product-wrap'>

			<div class='repeater-header row columns-4 span-6'>
				<div class='col'><label>
					<span class='label-text'><?php _e( 'Product', 'woocommerce-advanced-shipping-advanced-pricing' ); ?></span>
					<img class="help_tip" data-tip="<?php _e( 'Select a product to apply the fee amount to when the min/max quantity match.', 'woocommerce-advanced-shipping-advanced-pricing' ); ?>" src="<?php echo  WC()->plugin_url(); ?>/assets/images/help.png" height="16" width="16" />
				</label></div>
				<div class='col'><label>
					<span class='label-text'><?php _e( 'Min quantity', 'woocommerce-advanced-shipping-advanced-pricing' ); ?></span>
					<img class="help_tip" data-tip="<?php _e( 'You can set a minimum product quantity per row before the fee amount is applied.<br/>Leave empty to not set a minimum.', 'woocommerce-advanced-shipping-advanced-pricing' ); ?>" src="<?php echo  WC()->plugin_url(); ?>/assets/images/help.png" height="16" width="16" />
				</label></div>
				<div class='col'><label>
					<span class='label-text'><?php _e( 'Max quantity', 'woocommerce-advanced-shipping-advanced-pricing' ); ?></span>
					<img class="help_tip" data-tip="<?php _e( 'You can set a maximum product quantity per row before the fee amount is applied.<br/>Leave empty to not set a maximum.', 'woocommerce-advanced-shipping-advanced-pricing' ); ?>" src="<?php echo  WC()->plugin_url(); ?>/assets/images/help.png" height="16" width="16" />
				</label></div>
				<div class='col'><label>
					<span class='label-text'><?php _e( 'Fee amount', 'woocommerce-advanced-shipping-advanced-pricing' ); ?></span>
					<img class="help_tip" data-tip="<?php _e( 'Options are:<br/>5 : Flat amount<br/>5% : percentage of subtotal<br/>5%% : percentage of product total.<br/>Add a minus (-) to deduct.<br/>Add a asterisk (*) to multiply with quantity of the product.', 'woocommerce-advanced-shipping-advanced-pricing' ); ?>" src="<?php echo  WC()->plugin_url(); ?>/assets/images/help.png" height="16" width="16" />
				</label></div>

				<div class='col-inline'></div>
			</div>
			<div class='repeater-wrap'><?php

				$i = 0;
				if ( is_array( $pricing_rules ) ) :
					foreach ( $pricing_rules as $values ) :

						$i++;
						$product = isset( $values['condition']['product'] ) ? esc_attr( $values['condition']['product'] ) : '';
						$min     = isset( $values['condition']['min'] ) ? esc_attr( $values['condition']['min'] ) : '';
						$max     = isset( $values['condition']['max'] ) ? esc_attr( $values['condition']['max'] ) : '';
						$cost    = isset( $values['action']['cost'] ) ? esc_attr( $values['action']['cost'] ) : '';

						$_product = wc_get_product( $product );

						?><div class='cost-per-product-option repeater-row row columns-4 span-6'>

							<div class='col'>
								<select
									class='cost-per-product wc-product-search'
									name='advanced_pricing_<?php echo esc_attr( $this->id ); ?>[<?php echo absint( $i ); ?>][condition][product]'
									data-action="woocommerce_json_search_products_and_variations"
									placeholder="<?php _e( 'Select a product', 'woocommerce-advanced-shipping-advanced-pricing' ); ?>&hellip;"
									style="width: 100%;"
								><?php
								if ( $_product ) :
									?><option value="<?php echo absint( $product ); ?>"><?php echo $_product->get_formatted_name(); ?></option><?php
								endif;
								?></select>
							</div>
							<div class='col'>
								<input type='text' class='cost-per-product-min' name='advanced_pricing_<?php echo esc_attr( $this->id ); ?>[<?php echo absint( $i ); ?>][condition][min]' value='<?php echo $min; ?>'>
							</div>
							<div class='col'>
								<input type='text' class='cost-per-product-max' name='advanced_pricing_<?php echo esc_attr( $this->id ); ?>[<?php echo absint( $i ); ?>][condition][max]' value='<?php echo $max; ?>'>
							</div>
							<div class='col'>
								<input type='text' class='cost-per-product-cost wasap_input_price' name='advanced_pricing_<?php echo esc_attr( $this->id ); ?>[<?php echo absint( $i ); ?>][action][cost]' value='<?php echo esc_attr( wc_format_localized_price( $cost ) ); ?>' placeholder='<?php _e( 'Fixed amount or percentage', 'woocommerce-advanced-shipping-advanced-pricing' ); ?>'>
							</div>
							<div class='col-inline'>
								<span class='dashicons dashicons-no-alt delete-repeater-row' style='line-height: 29px;'></span>
							</div>

						</div><?php

					endforeach;
				else :

					?><div class='cost-per-product-option repeater-row row columns-4 span-6'>

						<div class='col'>
							<select
								class='cost-per-product wc-product-search'
								name='advanced_pricing_<?php echo esc_attr( $this->id ); ?>[<?php echo absint( $i ); ?>][condition][product]'
								data-action="woocommerce_json_search_products_and_variations"
								placeholder="<?php _e( 'Select a product', 'woocommerce-advanced-shipping-advanced-pricing' ); ?>&hellip;"
								data-selected=""
								style="width: 100%;"
							></select>
						</div>
						<div class='col'>
							<input type='text' class='cost-per-product-min' name='advanced_pricing_<?php echo esc_attr( $this->id ); ?>[0][condition][min]' value=''>
						</div>
						<div class='col'>
							<input type='text' class='cost-per-product-max' name='advanced_pricing_<?php echo esc_attr( $this->id ); ?>[0][condition][max]' value=''>
						</div>
						<div class='col'>
							<input type='text' class='cost-per-product-cost wasap_input_price' name='advanced_pricing_<?php echo esc_attr( $this->id ); ?>[0][action][cost]' value='' placeholder='<?php _e( 'Fixed amount or percentage', 'woocommerce-advanced-shipping-advanced-pricing' ); ?>'>
						</div>

						<div class='col-inline'></div>

					</div><?php

				endif;

			?></div>

			<a href='javascript:void(0);' class='button secondary-button add add-repeat-row'><?php _e( 'Add new', 'woocommerce-advanced-shipping-advanced-pricing' ); ?></a>

		</div><?php

		// Add new repeater row
		wc_enqueue_js( "
			jQuery( '.wasap-tabbed-settings #cost_per_product ' ).on( 'click', '.add-repeat-row', function() {

				var repeater_wrap = $( this ).prev( '.repeater-wrap' );
				var original = repeater_wrap.find( '.repeater-row' ).first();
				var original_selected = original.find( '.cost-per-product' ).select2( 'data' ) ? original.find( '.cost-per-product' ).select2( 'data' ).text : '';
				original.find( '.cost-per-product' ).attr( 'data-selected', original_selected );

				original.find( '.wc-product-search' ).select2( 'destroy' ).removeClass( 'enhanced' );
				var clone = original.clone();
				var repeater_index = repeater_wrap.find( '.repeater-row' ).length;
				repeater_index++;
				clone.find( '[name*=\"[condition][product]\"]' ).attr( 'name', 'advanced_pricing_cost_per_product[' + parseInt( repeater_index ) + '][condition][product]' ).val( '' ).attr( 'data-selected', '' );
				clone.find( '[name*=\"[condition][min]\"]' ).attr( 'name', 'advanced_pricing_cost_per_product[' + parseInt( repeater_index ) + '][condition][min]' ).val( '' );
				clone.find( '[name*=\"[condition][max]\"]' ).attr( 'name', 'advanced_pricing_cost_per_product[' + parseInt( repeater_index ) + '][condition][max]' ).val( '' );
				clone.find( '[name*=\"[action][cost]\"]' ).attr( 'name', 'advanced_pricing_cost_per_product[' + parseInt( repeater_index ) + '][action][cost]' ).val( '' );

				repeater_wrap.append( clone ).find( '.repeater-row' ).last().hide().slideDown( 'fast' );
				jQuery( document.body ).trigger( 'wc-enhanced-select-init' );

			});
		" );

	}


	/**
	 * Get Cost.
	 *
	 * Apply the advanced pricing cost to the shipping cost.
	 *
	 * @since 2.0.0
	 *
	 * @param  float $shipping_cost    Current shipping cost.
	 * @param  int   $shipping_rate_id ID of the shipping rate being calculated.
	 * @param  array $package          Shipping package being processed.
	 * @return float                   Modified shipping cost.
	 */
	public function apply_advanced_pricing_cost( $shipping_cost, $shipping_rate_id, $package ) {

		$pricing_rules = $this->get_pricing_rules( $shipping_rate_id );

		if ( is_array( $pricing_rules ) ) :
			foreach ( $pricing_rules as $values ) :

				$product = isset( $values['condition']['product'] ) ? esc_attr( $values['condition']['product'] ) : null;
				$product = apply_filters( 'wpml_object_id', $product, 'product', true );
				$min     = isset( $values['condition']['min'] ) ? esc_attr( $values['condition']['min'] ) : null;
				$max     = isset( $values['condition']['max'] ) ? esc_attr( $values['condition']['max'] ) : null;
				$cost    = isset( $values['action']['cost'] ) ? esc_attr( $values['action']['cost'] ) : null;

				$product_quantity     = $this->get_quantity( $package, $product );
				$product_subtotal     = $this->get_quantity( $package, $product, '$' );
				$min_compare_quantity = $this->get_quantity( $package, $product, $min );
				$max_compare_quantity = $this->get_quantity( $package, $product, $max );

				$min = str_replace( array( 'w', '$' ), '', $min );
				$max = str_replace( array( 'w', '$' ), '', $max );

				// Bail if cost is not set
				if ( empty( $cost ) ) :
					continue;
				endif;

				// Bail if none of this product is in the cart.
				if ( 0 == $product_quantity ) :
					continue;
				endif;

				// Bail if minimum is not set, or item qty is not met
				if ( is_null( $min ) || ( ! empty( $min ) && $min_compare_quantity < $min ) ) :
					continue;
				endif;

				// Bail if maximum is not set, or item qty is not met
				if ( is_null( $max ) || ( ! empty( $max ) && $max_compare_quantity > $max ) ) :
					continue;
				endif;

				$shipping_cost += wasap_apply_shipping_cost( $cost, $product_quantity, $product_subtotal, $min );

			endforeach;
		endif;

		return $shipping_cost;

	}


	/**
	 * Get related products.
	 *
	 * Get the related products from the cart where the cost should be applied to.
	 *
	 * @since 2.0.2
	 *
	 * @param  array  $package Full original shipping package data.
	 * @param  string $value   Set value for the advanced option.
	 * @return array           List of related products
	 */
	public function get_related_products( $package, $value = null ) {

		$related_products = array();

		foreach ( $package['contents'] as $cart_key => $item ) :

			$product_id   = $item['product_id'];
			$variation_id = ! empty( $item['variation_id'] ) ? $item['variation_id'] : null;

			if ( $value == $product_id || $value == $variation_id ) :
				$related_products[ $cart_key ] = $item;
			endif;

		endforeach;

		return $related_products;

	}


}
