<?php
if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * Class WASAP_Price_Settings.
 *
 * Add settings to the WooCommerce Advanced Shipping settings meta box.
 *
 * @class       WASAP_Price_Settings
 * @author     	Jeroen Sormani
 * @package		WooCommerce Advanced Shipping Advanced Pricing
 * @version		1.0.0
 */
class WASAP_Price_Settings {


	/**
	 * Constructor.
	 *
	 * @since 1.0.0
	 */
	public function __construct() {

		$this->load_files();

		// Initialize pricing options
		add_action( 'init', array( $this, 'load_pricing_options' ) );

		// Add advanced pricing fields
		add_action( 'was_after_meta_box_settings', array( $this, 'meta_box_settings_advanced_pricing' ) );

		// Save settings (legacy rate)
		add_action( 'was_save_shipping_settings', array( $this, 'save_advanced_price_settings' ) );

		// Save settings (WC Zone rate)
		add_filter( 'woocommerce_shipping_' . 'advanced_shipping' . '_instance_settings_values', array( $this, 'save_instance_settings' ), 10, 2 );

		// Add shipping costs.
		add_filter( 'was_calculate_shipping_costs', array( $this, 'add_advanced_pricing_shipping_costs' ), 10, 4 );
	}


	/**
	 * Load pricing files.
	 *
	 * Load the Advanced Pricing files.
	 *
	 * @since 1.1.0
	 */
	public function load_files() {

		// Load abstract
		require_once plugin_dir_path( __FILE__ ) . '/pricing-options/class-wasap-pricing-option-abstract.php';

		// Load core pricing options
		require_once plugin_dir_path( __FILE__ ) . '/pricing-options/class-wasap-pricing-option-cost-per-category.php';
		require_once plugin_dir_path( __FILE__ ) . '/pricing-options/class-wasap-pricing-option-cost-per-product.php';
		require_once plugin_dir_path( __FILE__ ) . '/pricing-options/class-wasap-pricing-option-cost-per-shipping-class.php';
		require_once plugin_dir_path( __FILE__ ) . '/pricing-options/class-wasap-pricing-option-cost-per-weight.php';

		// Allow 3rd party to load at this point
		do_action( 'wasap_load_pricing_option_files' );
	}


	/**
	 * Load pricing options.
	 *
	 * Auto load the pricing options.
	 *
	 * @since 1.1.0
	 * @since 2.0.0 Change hook from wasap_pricing_option to wasap_registered_pricing_options
	 *              Change how pricing options are loaded.
	 */
	public function load_pricing_options() {

		$registered_pricing_options = apply_filters( 'wasap_registered_pricing_options', array(
			'cost_per_weight',
			'cost_per_shipping_class',
			'cost_per_category',
			'cost_per_product',
		) );

		$pricing_options = array();
		foreach ( $registered_pricing_options as $pricing_option_type ) {

			// Get the correct class name
			if ( $class_name = $this->pricing_option_class_name_from_type( $pricing_option_type ) ) {
				$class                         = new $class_name;
				$pricing_options[ $class->id ] = $class;
			}

		}

		return $pricing_options;
	}


	/**
	 * Get pricing option class name.
	 *
	 * Get the class name of a cost option based on the cost type (ID).
	 *
	 * @since 2.0.0
	 *
	 * @param  string $type Pricing option type (same as ID).
	 * @return mixed        Pricing option class name when it exists, false otherwise.
	 */
	public function pricing_option_class_name_from_type( $type = '' ) {

		$class_name = 'WASAP_Pricing_Option_' . implode( '_', array_map( 'ucfirst', explode( '_', $type ) ) );

		if ( ! class_exists( $class_name ) ) {
			return false;
		}

		return $class_name;
	}


	/**
	 * Get pricing options.
	 *
	 * Get a list of the available pricing options. Alias function for $this->load_pricing_options();
	 *
	 * @since 2.0.0
	 *
	 * @return WASAP_Pricing_Option_Abstract[]
	 */
	public function get_pricing_options() {
		return $this->load_pricing_options();
	}


	/**
	 * Advanced pricing settings.
	 *
	 * Display all the advanced pricing settings. Per item.
	 *
	 * @since 1.0.0
	 *
	 * @param array $settings Array of all the existing settings and values.
	 */
	public function meta_box_settings_advanced_pricing( $settings ) {

		?><h3 style='padding-left:0;'><?php _e( 'Advanced Pricing', 'woocommerce-advanced-shipping-advanced-pricing' ); ?></h3>

		<div id='advanced-pricing-wrapper'><?php

			$pricing_options = $this->get_pricing_options();
			$first_option    = reset( $pricing_options );
			?><div class='wasap-tabbed-settings'>

				<div class='inside'>

					<div class='tabs-panels-wrap'>
						<div class='tabs'>
							<ul><?php
								/** @var WASAP_Pricing_Option_Abstract $pricing_option */
								foreach ( $pricing_options as $key => $pricing_option ) :
									$name = $pricing_option->name;
									?><li class='<?php echo $pricing_option->id == $first_option->id ? 'active' : ''; ?>'><a href='javascript:void(0);' data-target='<?php echo esc_attr( $key ); ?>'><?php echo esc_html( $name ); ?></a></li><?php
								endforeach;
							?></ul>
						</div>

						<div class='panels'><?php

							/** @var WASAP_Pricing_Option_Abstract $pricing_option */
							foreach ( $pricing_options as $pricing_option ) :
								?><div id='<?php echo esc_attr( $pricing_option->id ); ?>' class='panel' style='<?php echo $pricing_option->id != $first_option->id ? 'display: none;' : ''; ?>'><?php
									$pricing_option->output();
								?></div><?php
							endforeach;

						?></div>

						<div class='clear'></div>
					</div>

				</div>

			</div>

		</div><?php // #advanced-pricing-wrapper

	}


	/**
	 * Save advanced pricing.
	 *
	 * Save the advanced pricing fields in the database.
	 *
	 * @since 1.0.0
	 *
	 * @param  int $post_id ID of the shipping method that's is saved.
	 */
	public function save_advanced_price_settings( $post_id ) {
		foreach ( $this->get_pricing_options() as $option ) {
			$option->save_advanced_costs( $post_id );
		}
	}


	/**
	 * Save custom instance settings.
	 *
	 * Save the custom instance settings that are not directly registered as $this->instance_form_fields;
	 *
	 * @since 1.1.0
	 *
	 * @param  array                        $settings List of existing settings to be saved.
	 * @param  WAS_Advanced_Shipping_Method $class    A instance of this class returned through the filter.
	 * @return array                                  List of modified settings to be saved
	 */
	public function save_instance_settings( $settings, $class ) {
		$post_data = $class->get_post_data();

		// Save pricing options
		foreach ( $this->get_pricing_options() as $key => $option ) {
			$name       = 'advanced_pricing_' . esc_attr( $option->id );
			$save_value = isset( $post_data[ $name ] ) ? $post_data[ $name ] : array();

			array_walk_recursive( $save_value, 'sanitize_text_field' );
			$class->instance_settings[ $name ] = $save_value;
		}

		return $class->instance_settings;
	}


	/**
	 * Add shipping costs.
	 *
	 * Add shipping costs for this advanced pricing option.
	 *
	 * @since 1.1.0
	 *
	 * @param  double                       $cost             Current shipping cost.
	 * @param  array                        $package          Array of package details.
	 * @param  int                          $shipping_rate_id ID of the shipping rate.
	 * @param  WAS_Advanced_Shipping_Method $class            WAS class method.
	 * @return float                                          New shipping cost.
	 */
	public function add_advanced_pricing_shipping_costs( $cost, $package, $shipping_rate_id, $class ) {
		foreach ( $this->get_pricing_options() as $option ) {
			$cost = $option->apply_advanced_pricing_cost( $cost, $shipping_rate_id, $package );
		}

		return $cost;
	}
}
