<?php
if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * Class WASAP_Pricing_Option_Cost_Per_Shipping_Class.
 *
 * Cost per shipping class cost option.
 *
 * @class       WASAP_Pricing_Option_Cost_Per_Shipping_Class
 * @author     	Jeroen Sormani
 * @package		WooCommerce Advanced Shipping Advanced Pricing
 * @version		2.0.0
 */
class WASAP_Pricing_Option_Cost_Per_Shipping_Class extends WASAP_Pricing_Option_Abstract {


	/**
	 * Constructor.
	 *
	 * @since 2.0.0
	 */
	public function __construct() {

		$this->id   = 'cost_per_shipping_class';
		$this->name = __( 'Cost per shipping class', 'woocommerce-advanced-shipping-advanced-pricing' );

		parent::__construct();

	}


	/**
	 * Output settings.
	 *
	 * Output the settings related to this cost option.
	 *
	 * @since 2.0.0
	 */
	public function output() {

		$pricing_rules = $this->get_pricing_rules();

		?><div class='cost-per-shipping-class-wrap'>

			<div class='repeater-header row columns-4 span-6'>
				<div class='col'><label>
					<span class='label-text'><?php _e( 'Shipping class', 'woocommerce-advanced-shipping-advanced-pricing' ); ?></span>
					<img class="help_tip" data-tip="<?php _e( 'Select a shipping class to apply the fee amount to when the min/max quantity match.', 'woocommerce-advanced-shipping-advanced-pricing' ); ?>" src="<?php echo  WC()->plugin_url(); ?>/assets/images/help.png" height="16" width="16" />
				</label></div>
				<div class='col'><label>
					<span class='label-text'><?php _e( 'Min quantity', 'woocommerce-advanced-shipping-advanced-pricing' ); ?></span>
					<img class="help_tip" data-tip="<?php _e( 'You can set a minimum product quantity per row before the fee amount is applied.<br/>Leave empty to not set a minimum.', 'woocommerce-advanced-shipping-advanced-pricing' ); ?>" src="<?php echo  WC()->plugin_url(); ?>/assets/images/help.png" height="16" width="16" />
				</label></div>
				<div class='col'><label>
					<span class='label-text'><?php _e( 'Max quantity', 'woocommerce-advanced-shipping-advanced-pricing' ); ?></span>
					<img class="help_tip" data-tip="<?php _e( 'You can set a maximum product quantity per row before the fee amount is applied.<br/>Leave empty to not set a maximum.', 'woocommerce-advanced-shipping-advanced-pricing' ); ?>" src="<?php echo  WC()->plugin_url(); ?>/assets/images/help.png" height="16" width="16" />
				</label></div>
				<div class='col'><label>
					<span class='label-text'><?php _e( 'Fee amount', 'woocommerce-advanced-shipping-advanced-pricing' ); ?></span>
					<img class="help_tip" data-tip="<?php _e( 'Options are:<br/>5 : Flat amount<br/>5% : percentage of subtotal<br/>5%% : percentage of class total.<br/>Add a minus (-) to deduct.<br/>Add a asterisk (*) to multiply with quantity of the class.', 'woocommerce-advanced-shipping-advanced-pricing' ); ?>" src="<?php echo  WC()->plugin_url(); ?>/assets/images/help.png" height="16" width="16" />
				</label></div>

				<div class='col-inline'></div>
			</div>
			<div class='repeater-wrap'><?php

				$i = 0;
				if ( is_array( $pricing_rules ) ) :
					foreach ( $pricing_rules as $values ) :

						$i++;
						$shipping_class = isset( $values['condition']['shipping_class'] ) ? esc_attr( $values['condition']['shipping_class'] ) : '';
						$min            = isset( $values['condition']['min'] ) ? esc_attr( $values['condition']['min'] ) : '';
						$max            = isset( $values['condition']['max'] ) ? esc_attr( $values['condition']['max'] ) : '';
						$cost           = isset( $values['action']['cost'] ) ? esc_attr( $values['action']['cost'] ) : '';

						?><div class='cost-per-shipping-class-option repeater-row row columns-4 span-6'>

							<div class='col'>
								<select class='cost-per-shipping-class-shipping-class' name='advanced_pricing_<?php echo esc_attr( $this->id ); ?>[<?php echo absint( $i ); ?>][condition][shipping_class]'><?php
									?><option value=''><?php _e( 'No shipping class', 'woocommerce-advanced-shipping-advanced-pricing' ); ?></option><?php

									foreach ( get_terms( 'product_shipping_class', array( 'hide_empty' => false ) ) as $ship_class ) :
										?><option value='<?php echo esc_attr( $ship_class->slug ); ?>' <?php selected( $shipping_class, $ship_class->slug ); ?>><?php
											echo esc_attr( $ship_class->name );
										?></option><?php
									endforeach;
								?></select>
							</div>
							<div class='col'>
								<input type='text' class='cost-per-shipping-class-min' name='advanced_pricing_<?php echo esc_attr( $this->id ); ?>[<?php echo absint( $i ); ?>][condition][min]' value='<?php echo $min; ?>'>
							</div>
							<div class='col'>
								<input type='text' class='cost-per-shipping-class-max' name='advanced_pricing_<?php echo esc_attr( $this->id ); ?>[<?php echo absint( $i ); ?>][condition][max]' value='<?php echo $max; ?>'>
							</div>
							<div class='col'>
								<input type='text' class='cost-per-shipping-class-cost wasap_input_price' name='advanced_pricing_<?php echo esc_attr( $this->id ); ?>[<?php echo absint( $i ); ?>][action][cost]' value='<?php echo esc_attr( wc_format_localized_price( $cost ) ); ?>' placeholder='<?php _e( 'Fixed amount or percentage', 'woocommerce-advanced-shipping-advanced-pricing' ); ?>'>
							</div>
							<div class='col-inline'>
								<span class='dashicons dashicons-no-alt delete-repeater-row' style='line-height: 29px;'></span>
							</div>

						</div><?php

					endforeach;
				else :

					?><div class='cost-per-shipping-class-option repeater-row row columns-4 span-6'>
						<div class='col'>
							<select class='cost-per-shipping-class-shipping-class' name='advanced_pricing_<?php echo esc_attr( $this->id ); ?>[0][condition][shipping_class]'><?php
								?><option value=''><?php _e( 'No shipping class', 'woocommerce-advanced-shipping-advanced-pricing' ); ?></option><?php

								foreach ( get_terms( 'product_shipping_class', array( 'hide_empty' => false ) ) as $shipping_class ) :
									?><option value='<?php echo esc_attr( $shipping_class->slug ); ?>'><?php
										echo esc_attr( $shipping_class->name );
									?></option><?php
								endforeach;
							?></select>
						</div>
						<div class='col'>
							<input type='text' class='cost-per-shipping-class-min' name='advanced_pricing_<?php echo esc_attr( $this->id ); ?>[0][condition][min]' value=''>
						</div>
						<div class='col'>
							<input type='text' class='cost-per-shipping-class-max' name='advanced_pricing_<?php echo esc_attr( $this->id ); ?>[0][condition][max]' value=''>
						</div>
						<div class='col'>
							<input type='text' class='cost-per-shipping-class-cost wasap_input_price' name='advanced_pricing_<?php echo esc_attr( $this->id ); ?>[0][action][cost]' value='' placeholder='<?php _e( 'Fixed amount or percentage', 'woocommerce-advanced-shipping-advanced-pricing' ); ?>'>
						</div>

						<div class='col-inline'></div>

					</div><?php

				endif;

			?></div>

			<a href='javascript:void(0);' class='button secondary-button add add-repeat-row'><?php _e( 'Add new', 'woocommerce-advanced-shipping-advanced-pricing' ); ?></a>

		</div><?php

		// Add new repeater row
		wc_enqueue_js( "
			jQuery( '.wasap-tabbed-settings #cost_per_shipping_class' ).on( 'click', '.add-repeat-row', function() {

				var repeater_wrap = $( this ).prev( '.repeater-wrap' );
				var clone = repeater_wrap.find( '.repeater-row' ).first().clone();
				var repeater_index = repeater_wrap.find( '.repeater-row' ).length;
				repeater_index++;
				clone.find( '[name*=\"[condition][shipping_class]\"]' ).attr( 'name', 'advanced_pricing_cost_per_shipping_class[' + parseInt( repeater_index ) + '][condition][shipping_class]' ).val( '' );
				clone.find( '[name*=\"[condition][min]\"]' ).attr( 'name', 'advanced_pricing_cost_per_shipping_class[' + parseInt( repeater_index ) + '][condition][min]' ).val( '' );
				clone.find( '[name*=\"[condition][max]\"]' ).attr( 'name', 'advanced_pricing_cost_per_shipping_class[' + parseInt( repeater_index ) + '][condition][max]' ).val( '' );
				clone.find( '[name*=\"[action][cost]\"]' ).attr( 'name', 'advanced_pricing_cost_per_shipping_class[' + parseInt( repeater_index ) + '][action][cost]' ).val( '' );
				repeater_wrap.append( clone ).find( '.repeater-row' ).last().hide().slideDown( 'fast' );

			});
		" );

	}


	/**
	 * Get Cost.
	 *
	 * Apply the advanced pricing cost to the shipping cost.
	 *
	 * @since 2.0.0
	 *
	 * @param  float $shipping_cost    Current shipping cost.
	 * @param  int   $shipping_rate_id ID of the shipping rate being calculated.
	 * @param  array $package          Shipping package being processed.
	 * @return float                   Modified shipping cost.
	 */
	public function apply_advanced_pricing_cost( $shipping_cost, $shipping_rate_id, $package ) {

		$pricing_rules = $this->get_pricing_rules( $shipping_rate_id );

		// Loop through fee cost options
		if ( is_array( $pricing_rules ) ) :
			foreach ( $pricing_rules as $values ) :

				$shipping_class      = isset( $values['condition']['shipping_class'] ) ? esc_attr( $values['condition']['shipping_class'] ) : null;
				$shipping_class_term = get_term_by( 'slug', $shipping_class, 'product_shipping_class' );
				$shipping_class_id   = apply_filters( 'wpml_object_id', $shipping_class_term->term_id ?? null, 'product_shipping_class', true );

				$min  = isset( $values['condition']['min'] ) ? esc_attr( $values['condition']['min'] ) : null;
				$max  = isset( $values['condition']['max'] ) ? esc_attr( $values['condition']['max'] ) : null;
				$cost = isset( $values['action']['cost'] ) ? esc_attr( $values['action']['cost'] ) : null;

				$shipping_class_quantity = $this->get_quantity( $package, $shipping_class_id );
				$shipping_class_subtotal = $this->get_quantity( $package, $shipping_class_id, '$' );
				$min_compare_quantity    = $this->get_quantity( $package, $shipping_class_id, $min );
				$max_compare_quantity    = $this->get_quantity( $package, $shipping_class_id, $max );

				$min = str_replace( array( 'w', '$' ), '', $min );
				$max = str_replace( array( 'w', '$' ), '', $max );

				// Bail if cost is not set
				if ( empty( $cost ) ) :
					continue;
				endif;

				// Bail if shipping class is set
				if ( 0 == $shipping_class_quantity ) :
					continue;
				endif;

				// Bail if minimum is not set, or item qty is not met
				if ( is_null( $min ) || ( ! empty( $min ) && $min_compare_quantity < $min ) ) :
					continue;
				endif;

				// Bail if maximum is not set, or item qty is not met
				if ( is_null( $max ) || ( ! empty( $max ) && $max_compare_quantity > $max ) ) :
					continue;
				endif;

				$shipping_cost += wasap_apply_shipping_cost( $cost, $shipping_class_quantity, $shipping_class_subtotal, $min );

			endforeach;
		endif;

		return $shipping_cost;

	}


	/**
	 * Get related products.
	 *
	 * Get the related products from the cart where the cost should be applied to.
	 *
	 * @since 2.0.2
	 *
	 * @param  array  $package Full original shipping package data.
	 * @param  string $value   Set value for the advanced option.
	 * @return array           List of related products
	 */
	public function get_related_products( $package, $value = null ) {

		$related_products = array();

		foreach ( $package['contents'] as $cart_key => $item ) :

			$product_id          = ! empty( $item['variation_id'] ) ? $item['variation_id'] : $item['product_id'];
			$product             = wc_get_product( $product_id );
			$shipping_class      = $product->get_shipping_class();
			$shipping_class_term = get_term_by( 'slug' , $shipping_class, 'product_shipping_class' );
			$shipping_class_id   = apply_filters( 'wpml_object_id', $shipping_class_term->term_id ?? null, 'product_shipping_class', true );

			if ( $value == $shipping_class_id ) :
				$related_products[ $cart_key ] = $item;
			endif;

		endforeach;


		return $related_products;

	}


}
