<?php
if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * Class WASAP_Pricing_Option_Abstract.
 *
 * Abstract class to add new WASAP Cost Option.
 *
 * @class       WASAP_Pricing_Option_Abstract
 * @author     	Jeroen Sormani
 * @package		WooCommerce Advanced Shipping Advanced Pricing
 * @version		2.0.0
 */
abstract class WASAP_Pricing_Option_Abstract {


	/**
	 * Cost option ID.
	 *
	 * @since 2.0.0
	 * @var string $id ID of the cost option.
	 */
	public $id;


	/**
	 * Cost option name.
	 *
	 * @since 2.0.0
	 * @var string $name Name of the pricing option.
	 */
	public $name;


	/**
	 * Constructor.
	 *
	 * @since 2.0.0
	 */
	public function __construct() {

		// Save settings
		add_action( 'woocommerce_advanced_fees_save_meta_boxes', array( $this, 'save_advanced_costs' ) );

		// Register option.
		add_filter( 'woocommerce_advanced_fees_cost_options', array( $this, 'add_pricing_option' ) );

	}


	/**
	 * Add cost option.
	 *
	 * Add the cost option to the registered options.
	 *
	 * @since 2.0.0
	 *
	 * @param  array $options List of existing cost options.
	 * @return array          List of modified cost options.
	 */
	public function add_pricing_option( $options ) {

		$options[ $this->id ] = $this->name;

		return $options;

	}


	/**
	 * Get the pricing rules.
	 *
	 * Get the setup pricing rules for the current post.
	 *
	 * @since 1.0.0
	 *
	 * @param  int   $rate_id Post ID to get the pricing options for.
	 * @return mixed
	 */
	public function get_pricing_rules( $rate_id = null ) {

		// New WC Zones instance
		preg_match( '/advanced_shipping:([0-9]+)/', $rate_id ?? '', $matches );
		if ( isset( $_GET['instance_id'] ) || ! empty( $matches ) ) {
			$instance_id = absint( $_GET['instance_id'] ?? $matches[1] ?? 0 );

			if ( $shipping_method = WC_Shipping_Zones::get_shipping_method( $instance_id ) ) {
				return $shipping_method->get_instance_option( 'advanced_pricing_' . $this->id );
			}
		}

		// Legacy rate rules (used in admin)
		if ( is_null( $rate_id ) ) {
			$rate_id = get_the_ID();
		}

		return get_post_meta( $rate_id, 'advanced_pricing_' . esc_attr( $this->id ), true );
	}


	/**
	 * Output settings.
	 *
	 * Output the settings related to this cost option.
	 *
	 * @since 2.0.0
	 */
	abstract function output();


	/**
	 * Get Cost.
	 *
	 * Apply the advanced pricing cost to the shipping cost.
	 *
	 * @since 2.0.0
	 *
	 * @param  float $shipping_cost    Current shipping cost.
	 * @param  int   $shipping_rate_id ID of the shipping rate being calculated.
	 * @param  array $package          Shipping package being processed.
	 * @return float                   Modified shipping cost.
	 */
	abstract function apply_advanced_pricing_cost( $shipping_cost, $shipping_rate_id, $package );


	/**
	 * Save settings.
	 *
	 * Save the advanced cost settings.
	 *
	 * @since 2.0.0
	 *
	 * @param int $post_id ID of the post being saved.
	 */
	public function save_advanced_costs( $post_id ) {

		$name       = 'advanced_pricing_' . esc_attr( $this->id );
		$save_value = isset( $_POST[ $name ] ) ? $_POST[ $name ] : array();

		array_walk_recursive( $save_value, 'sanitize_text_field' );
		update_post_meta( $post_id, 'advanced_pricing_' . esc_attr( $this->id ), $save_value );

	}


	/**
	 * Get quantity.
	 *
	 * Get the quantity to compare the min/max values against. There are some
	 * special characters that can be used as listed below.
	 *
	 * - Use a 'w' in the min and/or max field to set the requirement based on weight of all the related products.
	 * - use a '$' sign in the min and/or max field to set the requirement based on subtotal of all the related products.
	 * - By default, the min/max field requirement is based on the quantity of the related products.
	 *
	 * @param       $package
	 * @param       $value
	 * @param  string $qty
	 * @return int
	 */
	public function get_quantity( $package, $value = null, $qty = '' ) {

		$quantity = 0;

		foreach ( $this->get_related_products( $package, $value ) as $cart_key => $item ) :

			if ( strpos( $qty, 'w' ) !== false ) :
				$quantity += $item['data']->get_weight() * $item['quantity'];
			elseif ( strpos( $qty, '$' ) !== false ) :
				$quantity += $item['data']->get_price() * $item['quantity'];
			else :
				$quantity += $item['quantity'];
			endif;

		endforeach;

		return $quantity;

	}


	/**
	 * Get related products.
	 *
	 * Get the related products from the cart where the cost should be applied to.
	 *
	 * @since 2.0.2
	 *
	 * @param  array  $package Full original shipping package data.
	 * @param  string $value   Set value for the advanced option.
	 * @return array           List of related products
	 */
	public function get_related_products( $package, $value = null ) {

		$related_products = array();

		foreach ( $package['contents'] as $cart_key => $item ) :
			$related_products[ $cart_key ] = $item;
		endforeach;

		return $related_products;

	}


}
