<?php

use JeroenSormani\WP_Updater\Plugin;
use function JeroenSormani\WP_Updater\wpu_get_template;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * Admin class.
 *
 * Handle all admin related functions.
 *
 * @author     	Jeroen Sormani
 * @version		1.0.0
 */
class WASAP_Admin {

	/**
	 * @var Plugin
	 */
	protected $plugin;

	/**
	 * Constructor.
	 *
	 * @since 1.0.5
	 */
	public function __construct() {
		// Auto updater function
		add_action( 'admin_init', array( $this, 'auto_updater' ), 5 );

		// Update license status on base plugin status update
		add_action( 'wp_updater\update_license_status', array( $this, 'update_status_on_base_plugin_update' ), 5, 2 );
	}



	/**
	 * Updater.
	 *
	 * Function to get automatic updates.
	 *
	 * @since NEWVERSION
	 */
	public function auto_updater() {

		// Updater
		if ( ! class_exists( '\JeroenSormani\WP_Updater\WPUpdater' ) ) {
			require plugin_dir_path( WooCommerce_Advanced_Shipping()->file ) . '/libraries/wp-updater/wp-updater.php';
		}
		$args = array(
			'file'                => WooCommerce_Advanced_Shipping_Advanced_Pricing()->file,
			'name'                => 'Advanced Shipping for WooCommerce - Advanced Pricing',
			'version'             => WooCommerce_Advanced_Shipping_Advanced_Pricing()->version,
			'api_url'             => 'https://aceplugins.com',
			'license_option_name' => 'woocommerce-advanced-shipping_license',
		);
		$updater = new \JeroenSormani\WP_Updater\WPUpdater( $args );
		$this->plugin = new Plugin( $args );

		// Remove default license field (used from base plugin)
		remove_action( 'after_plugin_row_' . plugin_basename( $args['file'] ), array( $updater, 'license_field' ), 20, 3 );

		// Show notice when update is available
		add_action( 'after_plugin_row_' . plugin_basename( $args['file'] ), array( $this, 'update_available_notice' ), 20, 3 );

	}

	/**
	 * Output notice.
	 *
	 * @since NEWVERSION
	 *
	 * @param $plugin_file
	 * @param $plugin_data
	 * @param $status
	 * @return void
	 */
	function update_available_notice( $plugin_file, $plugin_data, $status ) {

		$plugin = $this->plugin;

		$was_license_stauts = get_option( 'woocommerce-advanced-shipping_license_status', 'pending' );

		// Clear+update status and update info when it should be valid (base plugin is valid)
		if ( $plugin->get_license_status() == 'pending' && $was_license_stauts == 'valid' ) {
			$status = $this->plugin->client->get_status( true );
			$this->plugin->check_and_update_license_status();
			delete_site_transient( 'update_plugins' );  // clears WP plugins info transient to update the above
		}

		if ( $plugin->get_license_status() !== 'valid' && $plugin->client->is_update_available() ) {
			?><tr class="wp-updater-license-row <?php echo sanitize_html_class( $plugin->get_license_status() ); ?>">
				<td colspan="5"><?php
					?><div class="update-message notice inline notice-error notice-alt" style="margin: 10px 0 5px;">
						<p><?php echo sprintf( __( 'A new version of %s is available.' ), $plugin->get_name() ); ?>&nbsp;<strong><?php _e( 'Please enter a supported license key for the Advanced Shipping base plugin to receive this update.' ); ?></strong></p>
					</div><?php
				?></td>
			</tr><?php
		}
	}


	/**
	 * Update status on base update.
	 *
	 * Make sure the license status is the same as the base plugin.
	 *
	 * @since NEWVERSION
	 *
	 * @param $status
	 * @param $plugin
	 * @return void
	 */
	function update_status_on_base_plugin_update( $status, $plugin ) {
		if ( $plugin->get_slug() === 'advanced-shipping-for-woocommerce' ) {

			$option_prefix = dirname( plugin_basename( WooCommerce_Advanced_Shipping_Advanced_Pricing()->file ) );
			update_option( $option_prefix . '_license_status', sanitize_text_field( $status ) );

			if ( $status == 'valid' ) {
				$this->plugin->client->get_plugin_update_info_from_api( true ); // Re-get plugin data (download url etc)
			}
		}
	}
}
